package kareltherobot;

import java.util.*;
import java.io.*;

/**
 * <p>This is the heart of the Karel J Robot runtime. This class offers methods to manipulte 
 * the world by adding walls, adding or removing beepers or by adding or manipulate the
 * robots in the world. Though there can never be more than one instance of the World,
 * it is not implemented using static methods. This is because that would make it hard to
 * extend the functionality of the World. I used the Singleton pattern to assure that there
 * is at most one instance of a World. Using a real object however offers the possibitity
 * to use polymorphism for Worlds and for example implement a world as an RMI object.</p>
 * <p>A World consists of streets and avenues. Where streets are routes running from east to
 * west (horizontally) and avenues are running from south to north (vertically). Robots are
 * running along those routes and every turn will end on a crossrod (where a street and an avenue
 * cross). The position of a robot is described be the numbers of the street and avenue
 * which are crossing. The counting starts in the west-southmost corner (lower-left corner) with
 * the value 1.</p>
 * <p>There are also walls blocking streets or avenues. And beepers lying at crossroads. Also
 * a robot cannot be west (left) of the first avenue of north (under) of the first street.</p>
 */
public class World implements Directions
{

	static private World singletonWorld = null;
	static private boolean stateChanged = false;

	static
	{
		World world = new World();
		World.registerWorld( world );
	}
	
	/**
	 * Get the only World instance. If no instance yet exists, it creates a new
	 * instance of World. Subclasses can set the protected field <code>singletonWorld</code>
	 * so that an instance of the subclass will be used.
	 * @return the only World instance
	 */
	static public World getWorld()
	{
		if( singletonWorld == null )
		{
			singletonWorld = new World();
		}
		return singletonWorld;
	}

	static protected final boolean registerWorld( World world )
	{
		if( stateChanged )
			return false;

		singletonWorld = world;
		return true;
	}

	/**
	 * For internal use. The world maintains a list of World.Robots to keep track of
	 * all robots in the world. This class also offers the toString() method to
	 * receive a String representation of the Robot's state, which is used by the
	 * trace() method.
	 */
	static protected class Robot
	{
		public int id;
		public boolean on = true;
		public int avenue;
		public int street;
		public int beepers;
		public int direction;
		public String toString()
		{
			String result = "RobotId " + id + " at (street: " + street + ") (avenue: " + avenue + ") (beepers: ";
			if( beepers < 0 )
				result += "infinite";
			else
				result += beepers;
			result += ") (direction: ";
			if( direction == North )
				result += "North ";
			else if( direction == South )
				result += "South ";
			else if( direction == West )
				result += "West ";
			else if( direction == East )
				result += "East ";
			
			result += ") ";
			if( on )
				result += "on";
			else
				result += "off";
			return result;
		}
	}
	
	/**
	 * For internal use. Keeps track of the Beepers lying in the world. It is not
	 * used to represent Beepers kept by robots. There may never exist a Beepers
	 * object with less than one beeper.
	 */
	static protected class Beepers
	{
		public int avenue;
		public int street;
		public int count;
	}
	
	/**
	 * For internal use. Keeps track of the walls standing in the world. If isEastWest
	 * is true, the wall runs from east to west, thereby blocking an avenue. In this case the
	 * the avenue <code>avenue</code> is blocked and it cannot be crossed to move from
	 * street <code>street</code> the the one north of <code>street</code>. If isNorthSouth
	 * is true, a street is blocked respectively, which cannot be used to move from
	 * <code>avenue</code> to east of <code>avenue</code>.
	 */
	static protected class Wall
	{
		public boolean isEastWest;
		public boolean isNorthSouth;
		public int avenue;
		public int street;
	}
	
	protected class DelayThread extends Thread
	{
		volatile boolean paused = false;
		boolean doDelay = true;
		
		public DelayThread()
		{
		}

		public DelayThread( boolean doDelay )
		{
			this.doDelay = doDelay;
		}
		
		public void run()
		{
			if( doDelay )
			{
				try
				{
					Thread.sleep( delay );
					while( paused )
					{
						Thread.sleep( 50 );
					}
				}
				catch( InterruptedException exception )
				{
					exception.printStackTrace();
				}
			}
		}
		
		public void pause()
		{
			paused = true;
		}
		
		public void unPause()
		{
			paused = false;
		}
	}
	
	private boolean trace = true;
	private int nextRobotId = 0;
	protected Hashtable robots = new Hashtable();
	protected Vector walls = new Vector(), beepers = new Vector();
	protected int avenues = 15, streets = 10;
	protected int delay = 1000;
	protected DelayThread delayThread = new DelayThread( false );

	protected World()
	{
	}
	
	/**
	 * Used when the World's state changed and a repaint is needed.
	 */
	protected void update()
	{
	}
	
	/**
	 * Used when a Robot's state changed. If tracing is enabled, the Robot's state
	 * is printed to stanard out.
	 */
	protected void trace( Robot robot )
	{
		stateChanged = true;
		if( trace )
			System.out.println( robot );
	}

	/**
	 * Used to receive the Robot for an Id. This method also assures that there is a delay
	 * between Robot motions. This is mostly for displaying purposes in this class. A subclass
	 * could use this method to synchronize different Threads of Robots of differen
	 * processes using RMI, though.
	 */
	protected Robot getRobot( Integer robotId, boolean sync )
	{
		Robot result = (Robot) robots.get( robotId );
		if( result == null )
			throw new RuntimeException( "No such robot." );

		if( !result.on )
			throw new RuntimeException( "Robot is turned off." );

		if( !sync )
			return result;

		sync();
		return result;
	}
	
	protected void sync()
	{
		DelayThread delayThread = this.delayThread;
		try
		{
			delayThread.join();
		}
		catch( InterruptedException exception )
		{
			exception.printStackTrace();
		}
		synchronized( this )
		{
			if( this.delayThread == delayThread )
			{
				this.delayThread = new DelayThread();
				this.delayThread.start();
			}
		}
	}
	
// methods for handling the world
	public static void resetWalls()
	{
		getWorld().resetWallsInternal();
	}
	
	protected void resetWallsInternal()
	{
		walls = new Vector();
		update();
	}

	public static void resetBeepers()
	{
		getWorld().resetBeepersInternal();
	}
	
	protected void resetBeepersInternal()
	{
		beepers = new Vector();
		update();
	}

	public static void resetRobots()
	{
		getWorld().resetRobotsInternal();
	}
	
	protected void resetRobotsInternal()
	{
		robots = new Hashtable();
		nextRobotId = 0;
		update();
	}
	
	/**
	 * Adds <code>count</code> beepers the the crossroad denoted by <code>street</code>
	 * and <code>avenue</code>.
	 */
	static public void placeBeepers( int street, int avenue, int count )
	{
		getWorld().placeBeepersInternal( street, avenue, count );
	}
	 
	protected synchronized void placeBeepersInternal( int street, int avenue, int count )
	{
		Enumeration enum = beepers.elements();
		while( enum.hasMoreElements() )
		{
			Beepers current = (Beepers) enum.nextElement();
			if( ( current.avenue == avenue ) && ( current.street == street ) )
			{
				current.count += count;
				return;
			}
		}
		Beepers newBeepers = new Beepers();
		newBeepers.avenue = avenue;
		newBeepers.street = street;
		newBeepers.count = count;
		beepers.add( newBeepers );
		update();
	}
	
	/**
	 * Starting at the avenue <code>atAvenue</code> and continuing to the east, this method
	 * places <code>lengthTowardEast</code> walls north of the street <code>northOfStreet</code>.
	 */
	static public void placeEWWall( int northOfStreet, int atAvenue, int lengthTowardEast )
	{
		getWorld().placeEWWallInternal( northOfStreet, atAvenue, lengthTowardEast );
	}

	protected synchronized void placeEWWallInternal( int northOfStreet, int atAvenue, int lengthTowardEast )
	{
		for( ; lengthTowardEast > 0; lengthTowardEast--, atAvenue++ )
		{
			Wall wall = new Wall();
			wall.isEastWest = true;
			wall.avenue = atAvenue;
			wall.street = northOfStreet;
			walls.add( wall );
		}
		update();
	}

	/**
	 * starting at the street <code>atStreet</code> and continuing to the north, this method
	 * places <code>lengthTowardNorth</code> walls east of the avenue <code>eastOfAvenue</code>.
	 */
	static public void placeNSWall( int atStreet, int eastOfAvenue, int lengthTowardNorth )
	{
		getWorld().placeNSWallInternal( atStreet, eastOfAvenue, lengthTowardNorth );
	}
	
	protected synchronized void placeNSWallInternal( int atStreet, int eastOfAvenue, int lengthTowardNorth )
	{
		for( ; lengthTowardNorth > 0; lengthTowardNorth--, atStreet++ )
		{
			Wall wall = new Wall();
			wall.isNorthSouth = true;
			wall.avenue = eastOfAvenue;
			wall.street = atStreet;
			walls.add( wall );
		}
		update();
	}

	/**
	 * This method sets the size of the visible World. At the lower-left corner there
	 * is alway displayed the crossing of street 1 and avenue 1. Horizontally <code>avenues</code>
	 * avenues are displayed and vertically <code>streets</code> streets are displayed.
	 * However beepers, walls and robots can also be (and move) further north or east of the displayes
	 * routes. Though they cannot be further west or south of the first avenue or street.
	 */
	public static void setSize( int streets, int avenues )
	{
		getWorld().setSizeInternal( streets, avenues );
	}

	protected synchronized void setSizeInternal( int streets, int avenues )
	{
		setAvenues( avenues );
		setStreets( streets );
		update();
	}

	public static void setStreets( int streets )
	{
		getWorld().setStreetsInternal( streets );
	}

	protected synchronized void setStreetsInternal( int streets )
	{
		this.streets = streets;
		if( this.streets < 1 )
			this.streets = 1;
		update();
	}

	public static void setAvenues( int avenues )
	{
		getWorld().setAvenuesInternal( avenues );
	}

	protected synchronized void setAvenuesInternal( int avenues )
	{
		this.avenues = avenues;
		if( this.avenues < 1 )
			this.avenues = 1;
		update();
	}

	/**
	 * Sets the delay between Robot movements. The delay is measured in milliseconds.
	 */
	public static void setDelay( int delay )
	{
		getWorld().setDelayInternal( delay );
	}

	protected void setDelayInternal( int delay )
	{
		this.delay = delay;
		update();
	}
	
	public static int delay()
	{
		return getWorld().delayInternal();
	}
	
	protected int delayInternal()
	{
		return delay;
	}
	
	/**
	 * Checks if there is at least one beeper at the crossing of street <code>street</code>
	 * and avenue <code>avenue</code>.
	 */
	public static boolean checkBeeper( int street, int avenue )
	{
		return getWorld().checkBeeperInternal( street, avenue );
	}
	
	protected boolean checkBeeperInternal( int street, int avenue )
	{
		Enumeration enum = beepers.elements();
		while( enum.hasMoreElements() )
		{
			Beepers currentBeepers = (Beepers) enum.nextElement();
			if( ( currentBeepers.street == street ) && ( currentBeepers.avenue == avenue ) )
				return true;
		}
		return false;
	}
	
	/**
	 * Checks if there is a wall east of the avenue <code>eastOfAvenue</code> crossing the
	 * street <code>atStreet</code>.
	 */
	public static boolean checkEWWall( int atStreet, int eastOfAvenue )
	{
		return getWorld().checkEWWallInternal( atStreet, eastOfAvenue );
	}
	
	protected boolean checkEWWallInternal( int atStreet, int eastOfAvenue )
	{
		Enumeration enum = walls.elements();
		while( enum.hasMoreElements() )
		{
			Wall currentWall = (Wall) enum.nextElement();
			if( currentWall.isEastWest && ( currentWall.street == atStreet ) && ( currentWall.avenue == eastOfAvenue ) )
				return true;
		}
		return false;
	}

	/**
	 * Checks if there is a wall north of the street <code>northOfStreet</code> crossing the
	 * avenue <code>atAvenue</code>.
	 */
	public static boolean checkNSWall( int northOfStreet, int atAvenue )
	{
		return getWorld().checkNSWallInternal( northOfStreet, atAvenue );
	}
	
	protected boolean checkNSWallInternal( int northOfStreet, int atAvenue )
	{
		Enumeration enum = walls.elements();
		while( enum.hasMoreElements() )
		{
			Wall currentWall = (Wall) enum.nextElement();
			if( currentWall.isNorthSouth && ( currentWall.street == northOfStreet ) && ( currentWall.avenue == atAvenue ) )
				return true;
		}
		return false;
	}

	/**
	 * Saves the current constellation of walls, beepers and visible size of the World
	 * into a file.
	 */
	public static void saveWorld( String filename )
	{
		getWorld().saveWorldInternal( filename );
	}
	
	protected synchronized void saveWorldInternal( String filename )
	{
		PrintWriter writer = null;
		try
		{
			writer = new PrintWriter( new FileWriter( filename ) );
			writer.println( "KarelWorld" );
			
			writer.println( "streets " + numberOfStreets() );
			writer.println( "avenues " + numberOfAvenues() );
			
			Enumeration enum = beepers.elements();
			while( enum.hasMoreElements() )
			{
				Beepers current = (Beepers) enum.nextElement();
				writer.println( "beepers " + current.street + " " + current.avenue + " " + current.count );
			}

			enum = walls.elements();
			while( enum.hasMoreElements() )
			{
				Wall current = (Wall) enum.nextElement();
				if( current.isEastWest )
					writer.println( "eastwestwalls " + current.street + " " + current.avenue + " " + current.avenue );
				else
					writer.println( "northsouthwalls " + current.avenue + " " + current.street + " " + current.street );
			}
		}
		catch( IOException exception )
		{
			exception.printStackTrace();
			System.out.println( "World no saved" );
		}
		finally
		{
			if( writer != null )
			{
				writer.flush();
				writer.close();
			}
		}
	}
	
	/**
	 * Reads the visible size of the World and the positions of walls and beepers from
	 * a file. The World is reset and except for the Robots and robot Ids. Then the
	 * walls and beepers read from the file are set into the world.
	 */
	public static void readWorld( String filename )
	{
		getWorld().readWorldInternal( filename );
	}

	protected synchronized void readWorldInternal( String filename )
	{
		resetWalls();
		resetBeepers();
		BufferedReader reader = null;
		try
		{
			reader = new BufferedReader( new FileReader( filename ) );
			while( reader.ready() )
			{
				StringTokenizer tokens = new StringTokenizer( reader.readLine(), " ", false );
				if( tokens.hasMoreTokens() )
				{
					String action = tokens.nextToken();
					if( action.equalsIgnoreCase( "streets" ) )
					{
						setStreets( Integer.parseInt( tokens.nextToken() ) );
					}
					else if( action.equalsIgnoreCase( "avenues" ) )
					{
						setAvenues( Integer.parseInt( tokens.nextToken() ) );
					}
					else if( action.equalsIgnoreCase( "beepers" ) )
					{
						placeBeepers( Integer.parseInt( tokens.nextToken() ),
							Integer.parseInt( tokens.nextToken() ),
							Integer.parseInt( tokens.nextToken() )
						);
					}
					else if( action.equalsIgnoreCase( "northsouthwalls" ) )
					{
						int eoa = Integer.parseInt( tokens.nextToken() );
						int fs = Integer.parseInt( tokens.nextToken() );
						int ls = Integer.parseInt( tokens.nextToken() );
						placeNSWall( fs, eoa, ls - fs + 1 );
					}
					else if( action.equalsIgnoreCase( "eastwestwalls" ) )
					{
						int nos = Integer.parseInt( tokens.nextToken() );
						int fa = Integer.parseInt( tokens.nextToken() );
						int la = Integer.parseInt( tokens.nextToken() );
						placeEWWall( nos, fa, la - fa + 1 );
					}
				}
			}
		}
		catch( IOException exception )
		{
			exception.printStackTrace();
			System.out.println( "No World read" );
		}
		finally
		{
			if( reader != null )
				try{ reader.close(); } catch( IOException exception ) {}
		}
	}

	/**
	 * Returns the number of streets that are currently displayed.
	 */
	public static int numberOfStreets()
	{
		return getWorld().numberOfStreetsInternal();
	}
	
	protected int numberOfStreetsInternal()
	{
		return streets;
	}

	/**
	 * Returns the numer of avenues that are currently displayed.
	 */
	public static int numberOfAvenues()
	{
		return getWorld().numberOfAvenuesInternal();
	}
	
	protected int numberOfAvenuesInternal()
	{
		return avenues;
	}

	/**
	 * Sets if Robot states are to be traced (true is the default). See <a href="#trace()">trace()</a>
	 * for details on tracing.
	 */
	public static void setTrace( boolean trace )
	{
		getWorld().setTraceInternal( trace );
	}

	protected void setTraceInternal( boolean trace )
	{
		this.trace = trace;
		update();
	}

// methods for handling robots
	/**
	 * This method is to be used to create a new Robot in this World.
	 * It should be called by the ur_Robot class which then is only
	 * a proxy. Every movement or state-change of the robot is actually
	 * done by the World.
	 */
	synchronized Integer getNewRobot( int street, int avenue, int direction, int beepers )
	{
		Integer currentRobotId = new Integer( nextRobotId );
		nextRobotId++;
		Robot robot = new Robot();
		robot.id = nextRobotId - 1;
		robot.avenue = avenue;
		robot.street = street;
		robot.beepers = beepers;
		robot.direction = direction;
		robots.put( currentRobotId, robot );
		update();
		trace( robot );
		return currentRobotId;
	}

	/**
	 * Returns the avenue on which the denoted Robot is currently standing.
	 */
	int getAvenue( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		return current.avenue;
	}

	/**
	 * Returns the street on which the denoted Robot is currently standing.
	 */
	int getStreet( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		return current.street;
	}

	/**
	 * Returns the current direction of the denoted Robot.
	 */
	int getDirection( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		return current.direction;
	}
	
	int getBeepers( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		return current.beepers;
	}
	
	boolean frontIsClear( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		Enumeration enum = walls.elements();
		while( enum.hasMoreElements() )
		{
			Wall currentWall = (Wall) enum.nextElement();
			if( current.direction == North )
			{
				if( currentWall.avenue == current.avenue && currentWall.street == current.street )
					return false;
			}
			else if( current.direction == South )
			{
				if( currentWall.avenue == current.avenue && currentWall.street == ( current.street - 1 ) )
					return false;
			}
			else if( current.direction == East )
			{
				if( currentWall.street == current.street && currentWall.avenue == current.avenue )
					return false;
			}
			else if( current.direction == West )
			{
				if( currentWall.street == ( current.street - 1 ) && currentWall.avenue == current.avenue )
					return false;
			}
		}
		return true;
	}

	boolean nextToARobot( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		Enumeration enum = robots.elements();
		while( enum.hasMoreElements() )
		{
			Robot currentRobot = (Robot) enum.nextElement();
			if( current.avenue == currentRobot.avenue && current.street == currentRobot.street && current != currentRobot )
				return true;
		}
		return false;
	}

	boolean nextToABeeper( Integer robotId )
	{
		Robot current = getRobot( robotId, false );

		Enumeration enum = beepers.elements();
		while( enum.hasMoreElements() )
		{
			Beepers currentBeepers = (Beepers) enum.nextElement();
			if( current.avenue == currentBeepers.avenue && current.street == currentBeepers.street )
				return true;
		}
		return false;
	}

	/**
	 * Move the denoted Robot one step (to the next crossing) in its direction.
	 * If the route it would be using is blocked, this method throws a RuntimeException.
	 */
	void move( Integer robotId )
	{
		Robot current = getRobot( robotId, true );

		Enumeration enum = walls.elements();
		switch( current.direction )
		{
			case North :
				while( enum.hasMoreElements() )
				{
					Wall currentWall = (Wall) enum.nextElement();
					if( ( currentWall.street == current.street ) && ( currentWall.avenue == current.avenue) && currentWall.isEastWest )
						throw new RuntimeException( "Cannot move through wall." );
				}
				current.street++;
				break;
			case South :
				while( enum.hasMoreElements() )
				{
					Wall currentWall = (Wall) enum.nextElement();
					if( ( currentWall.street == current.street - 1 ) && ( currentWall.avenue == current.avenue ) && currentWall.isEastWest )
						throw new RuntimeException( "Cannot move through wall." );
				}
				if( current.street == 1 )
					throw new RuntimeException( "Cannot move through wall." );
				current.street--;
				break;
			case West :
				while( enum.hasMoreElements() )
				{
					Wall currentWall = (Wall) enum.nextElement();
					if( ( currentWall.avenue == current.avenue - 1 ) && ( currentWall.street == current.street) && currentWall.isNorthSouth )
						throw new RuntimeException( "Cannot move through wall." );
				}
				if( current.avenue == 1 )
					throw new RuntimeException( "Cannot move through wall." );
				current.avenue--;
				break;
			case East :
				while( enum.hasMoreElements() )
				{
					Wall currentWall = (Wall) enum.nextElement();
					if( ( currentWall.avenue == current.avenue ) && ( currentWall.street == current.street ) && currentWall.isNorthSouth )
						throw new RuntimeException( "Cannot move through wall." );
				}
				current.avenue++;
				break;
		}
		trace( current );
		update();
	}
	
	/**
	 * Turns the denoted Robot left by 90 degrees.
	 */
	void turnLeft( Integer robotId )
	{
		Robot current = getRobot( robotId, true );

		switch( current.direction )
		{
			case North :
				current.direction = West;
				break;
			case South :
				current.direction = East;
				break;
			case West :
				current.direction = South;
				break;
			case East :
				current.direction = North;
				break;
		}
		trace( current );
		update();
	}

	/**
	 * Picks up a beeper and puts it into the denoted Robot's beeperbag, if the Robot
	 * stands on a crossroad where there is also at least one beeper. If no beeper
	 * is present, this method throws a RuntimeException.
	 */
	void pickBeeper( Integer robotId )
	{
		Robot current = getRobot( robotId, true );
		
		synchronized( this )
		{
			Enumeration enum = beepers.elements();
			while( enum.hasMoreElements() )
			{
				Beepers currentBeepers = (Beepers) enum.nextElement();
				if( ( currentBeepers.street == current.street ) && ( currentBeepers.avenue == current.avenue ) )
				{
					if( currentBeepers.count == 1 )
						beepers.remove( currentBeepers );
					else
						currentBeepers.count--;
					
					if( current.beepers >= 0 )
						current.beepers++;
					trace( current );
					update();
					return;
				}
			}
		}
		throw new RuntimeException( "No beepers to pick up." );
	}
	
	/**
	 * Puts a beeper from the denoted Robot's beeperbag onto the crossroad where it is#
	 * standing. If the Robot has no beepers, a RuntimeException is thrown.
	 */
	void putBeeper( Integer robotId )
	{
		Robot current = getRobot( robotId, true );
		
		if( current.beepers == 0 )
			throw new RuntimeException( "Has no beepers." );
		
		if( current.beepers > 0 )
			current.beepers--;
		placeBeepers( current.street, current.avenue, 1 );
		trace( current );
		update();
	}
	
	/**
	 * Turns of the denoted Robot.
	 */
	void turnOff( Integer robotId )
	{
		Robot current = getRobot( robotId, true );
		current.on = false;
		trace( current );
		update();
	}

// administrative methods

	public static void main( String args[] ) throws Exception
	{
		World world = World.getWorld();
		world.setSize( 10, 10 );
		world.placeBeepers( 4, 4, 1 );
		world.placeNSWall( 1, 1, 3 );
		Integer rob = world.getNewRobot( 4, 1, West, infinity );
		world.putBeeper( rob );
		world.turnLeft( rob ); world.turnLeft( rob );
		world.move( rob ); world.move( rob ); world.move( rob );
		world.pickBeeper( rob );
		world.move( rob );
	}
}