package kareltherobot;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.security.*;
import java.net.URL;
import java.util.*;

public class GUIWorld extends World
{

	private ControlExit security;
	private JFrame jFrame;
	private JTextField streetsText, avenuesText, delayText;
	private JButton pauseButton;
	
	static
	{
		GUIWorld world = new GUIWorld();
		if( World.registerWorld( world ) )
			world.init();
	}
	
	private GUIWorld()
	{
	}
	
	/**
	 * As the world opens a JFrame, the Virtual Machine shall only exit if the JFrame is closed.
	 * This is achieved using this SecurityManager which prohibits exiting the VM until
	 * the application has permitted it.
	 */
	private class ControlExit extends SecurityManager
	{
		boolean allowExit = false;
	
		/** Allow to exit the VM. */
		public void allowExit()
		{
			allowExit = true;
		}
		public void checkExit( int level )
		{
			if( !allowExit )
			{
				System.out.println("Forbidden VM exit suppressed.");
				throw new SecurityException();
			}
		}
		
		public void checkPermission( Permission perm )
		{
		}
		
		public void checkPermission( Permission perm, Object context )
		{
		}
	}
	
	/**
	 * This class is used to paint the World into a JFrame.
	 */
	private class SwingWorld extends JComponent
	{
		private Image crossroad, wallImgs[] = new Image[4], beepersImgs[] = new Image[10], karel[] = new Image[4];

		public SwingWorld()
		{
			// Load the gif images for painting the world. Use the Class.getResource() method. So
			// the images are loaded using the classloader and can, p.e. be in a jar archive.
			crossroad = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "crossroad.gif" ) );
			beepersImgs[0] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "beepersx.gif" ) );
			for( int i = 1; i <= 9; i++ )
				beepersImgs[i] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "beepers" + i + ".gif" ) );

			karel[North] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "karelnorth.gif" ) );
			karel[East] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "kareleast.gif" ) );
			karel[South] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "karelsouth.gif" ) );
			karel[West] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "karelwest.gif" ) );

			wallImgs[North] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "wallnorth.gif" ) );
			wallImgs[East] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "walleast.gif" ) );
			wallImgs[South] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "wallsouth.gif" ) );
			wallImgs[West] = Toolkit.getDefaultToolkit().createImage( World.class.getResource( "wallwest.gif" ) );			
		}
		
		public void paint( Graphics g )
		{
			int avDist = getWidth() / ( avenues - 1 );
			int stDist = getHeight() / ( streets - 1 );
			for( int a = 0; a < avenues - 1; a++ )
				for( int s = 0; s < streets - 1; s++ )
					drawImage( g, crossroad, a, s, avDist, stDist );

			for( int a = 0; a < avenues; a++ )
				drawImage( g, wallImgs[South], a, 0, avDist, stDist );

			for( int s = 0; s < streets; s++ )
				drawImage( g, wallImgs[West], 0, s, avDist, stDist );
					 
			Enumeration enum = walls.elements();
			while( enum.hasMoreElements() )
			{
				Wall current = (Wall) enum.nextElement();
				if( current.isEastWest )
					drawImage( g, wallImgs[North], current.avenue - 1, current.street - 1, avDist, stDist );
				else if( current.isNorthSouth )
					drawImage( g, wallImgs[East], current.avenue - 1, current.street - 1, avDist, stDist );
			}
			
			enum = beepers.elements();
			while( enum.hasMoreElements() )
			{
				Beepers current = (Beepers) enum.nextElement();
				int index = current.count;
				if( current.count > 9 )
					index = 0;
				drawImage( g, beepersImgs[index], current.avenue - 1, current.street - 1, avDist, stDist );
			}
			
			enum = robots.elements();
			while( enum.hasMoreElements() )
			{
				Robot current = (Robot) enum.nextElement();
				drawImage( g, karel[current.direction], current.avenue - 1, current.street - 1, avDist, stDist );
			}
			Toolkit.getDefaultToolkit().sync();
		}

		private void drawImage( Graphics g, Image image, int x, int y, int width, int height )
		{
			g.drawImage( image, x * width, getHeight() -  ( y + 1 ) * height, width, height, this );
		}
	}

	/**
	 * This method is called when the JFrame, which is the World, is closed. Only then
	 * allow the VM to be exited and quit the application.
	 */
	private void exitForm( java.awt.event.WindowEvent evt )
	{
		security.allowExit();
		System.exit (0);
	}
	
	private void init()
	{

		security = new ControlExit();
		System.setSecurityManager( security );

		jFrame = new JFrame();
	        jFrame.getContentPane().setLayout( new BorderLayout( 4, 4 ) );
		jFrame.getContentPane().add( new SwingWorld(), BorderLayout.CENTER );
		jFrame.getContentPane().add( getSizePanel(), BorderLayout.NORTH );
		jFrame.getContentPane().add( getDelayPanel(), BorderLayout.SOUTH );

		jFrame.setForeground( Color.black );
		jFrame.setBackground( Color.lightGray );

		jFrame.setDefaultCloseOperation( WindowConstants.DISPOSE_ON_CLOSE );
		jFrame.addWindowListener(new java.awt.event.WindowAdapter() {
				public void windowClosing(java.awt.event.WindowEvent evt) {
					exitForm( evt ); }
			}
		);
		
		Dimension ScreenSize = Toolkit.getDefaultToolkit().getScreenSize();
		int height =  ScreenSize.height * 3 / 4;
		int width = ScreenSize.width * 3 / 4;
		int x =  ScreenSize.width / 8;
		int y = ScreenSize.height / 8;
		jFrame.show();
		jFrame.setSize( new Dimension( width, height ) );
		jFrame.setLocation( x, y );
		jFrame.show();
		update();
	}
	
	protected JComponent getSizePanel()
	{
		JPanel result = new JPanel( new GridLayout( 1, 4, 4, 4 ) );
		result.add( new JLabel( "streets:" ) );
		streetsText = new JTextField( "" + numberOfStreets() );
		streetsText.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent event )
			{
				try
				{
					setStreets( Integer.parseInt( streetsText.getText() ) );
				}
				catch( NumberFormatException exception )
				{
				}
			}
		} );
		result.add( streetsText );
		result.add( new JLabel( "avenues:" ) );
		avenuesText = new JTextField( "" + numberOfAvenues() );
		avenuesText.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent event )
			{
				try
				{
					setAvenues( Integer.parseInt( avenuesText.getText() ) );
				}
				catch( NumberFormatException exception )
				{
				}
			}
		} );
		result.add( avenuesText );
		return result;
	}
	
	protected JComponent getDelayPanel()
	{
		JPanel result = new JPanel( new GridLayout( 1, 3, 4, 4 ) );
		result.add( new JLabel( "delay [ms]:" ) );
		delayText = new JTextField( "" + delay );
		delayText.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent event )
			{
				try
				{
					setDelay( Integer.parseInt( delayText.getText() ) );
				}
				catch( NumberFormatException exception )
				{
				}
			}
		} );
		result.add( delayText );
		pauseButton = new JButton( "pause" );
		pauseButton.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent event )
			{
				if( pauseButton.getText().equals( "pause" ) )
				{
					pauseButton.setText( "resume" );
					delayThread.pause();
				}
				else if( pauseButton.getText().equals( "resume" ) )
				{
					pauseButton.setText( "pause" );
					delayThread.unPause();
				}
			}
		} );
		result.add( pauseButton );
		return result;
	}
	
	protected void update()
	{
		if( jFrame != null )
			jFrame.repaint();
	}
	
	protected void setStreetsInternal( int streets )
	{
		super.setStreetsInternal( streets );
		if( streetsText != null )
			streetsText.setText( "" + streets );
	}
	
	protected void setAvenuesInternal( int avenues )
	{
		super.setAvenuesInternal( avenues );
		if( avenuesText != null )
			avenuesText.setText( "" + avenues );
	}

	protected void setDelayInternal( int delay )
	{
		super.setDelayInternal( delay );
		if( delayText != null )
			delayText.setText( "" + delay );
	}
}